<?php
/**
 * m'Manager main class
 * @package m'Manager Invoices Management System
 * @link https://codecanyon.net/item/mmanager-invoices-management-system/19866435
 * @author Eric Claver AKAFFOU (akaffou.eric@me.com)
 * @since Version 1.0
 */
defined('ONERIC_VERSION') or define('ONERIC_VERSION', get_option('stable_version'));
defined('USERID') or define('USERID', get_user_id());

use Swap\Builder;
use Mmanager\Version;
use PhpRbac\Rbac;
use Mmanager\Extensions\Woocommerce\WooCommerce;

class M_manager {

	// User properties
	protected $_userID;
	protected $_userEmail;
	protected $_userFname;
	protected $_userCompany;
	protected $_userOptions = [];
	protected $_userAddress = [];
	protected $user_language;
	// Options properties
	protected $_optionsID = [];
	protected $salt;
	// Client properties
	protected $client_id;
	// Invoices properties
	protected $invoice_array = [];
	protected $invoice_number;
	protected $invoice_status;
	protected $subject;
	protected $pay_type;
	protected $date;
	protected $currency;
	protected $due_date;
	protected $next_due_date;
	protected $default_due_date;
	protected $subtotal;
	protected $discount_amount;
	protected $tax_amount;
	protected $total;
	protected $change_rate;
	protected $amount_paid;
	protected $amount_due;
	protected $amount_refunded;
	protected $notes;
	protected $sent;
	protected $transaction_id;
	// Quote properties
	protected $quote_array = [];
	protected $quote_number;
	protected $quote_status;

	public function __construct()
	{
		$this->user_language = null !== get_option('user_language', 'users_options') ? get_option('user_language', 'users_options'): language_code_to_string(get_option('default_language'));
		// Load language file
		$this->lang->load('application', $this->user_language);
		// Load Ion_Auth language
		$this->lang->load('users/auth', $this->user_language);
		$this->lang->load('users/ion_auth', $this->user_language);

		$version = new Version;

		log_message('info', 'M_manager Class Initialized');
		if ( floatval(get_option('stable_version')) < 1.8 )
		{
			$this->load->helper('upgrade_db_to_1_8');
			db_change_quote_quantity();
			upgrade_db_to_1_8();
		}
		db_add_credit_note();
		db_add_accounts_counters();
		upgrade_to_additional_taxes();
		upgrade_dbs_for_additional_taxes();
		db_add_client_cc();
		$this->load->helper('upgrade_db_to_193');
		upgrade_db_to_193();

		// Add Php Rbac support to manage user roles and permission
		$this->load->helper('upgrade_db_to_php_rbac');
		upgrade_db_to_php_rbac();
		
		$this->config->set_item('time_reference', get_option('user_timezone'));
		$this->load->library('ezsql_codeigniter');
		$this->load->library('parser');
		$this->oneric = new ezsql_codeigniter;
		$this->config->load('db_tables');
		$this->tables = $this->config->item('tables');
		$this->salt = $this->config->item('salt');
		$this->_setUserVars();
		$this->_setOption();
		$this->_init();
	}
	public function __get($var)
	{
		return get_instance()->$var;
	}
	public function __call($method, $args)
	{}
	/**
	 * Save users options in oc_usersoptions 
	 * @param type $db_data 
	 * @return type mixed
	 */
	public function registerOption($db_data)
	{
		$this->load->model('settings/settings_model', 'settings');
		$this->settings->set_table('users_options');

		return $this->settings->save($db_data);
	}
	/**
	 * Get option_id based on it's name
	 * @param type $option_key 
	 * @return type
	 */
	public function getOptionID($option_key)
	{
		return array_key_exists($option_key, $this->_optionsID) ? $this->_optionsID[$option_key] : FALSE;
	}
	/**
	 * Fetchs the option_key in the data provided
	 * and update the option in oc_usersoptions
	 * @param type $db_data 
	 * @return type mixed
	 */
	public function updateOption($db_data)
	{
		$this->load->model('settings/settings_model', 'settings');
		$this->settings->set_table('users_options');
		$this->settings->set_primary_key('option_id');
		$option_key = $db_data['option_key'];
		$option_id = $this->getOptionID($option_key);

		return $this->settings->save($db_data, $option_id);
	}
	/**
	 * Get the next invoice id
	 * @return type int
	 */
	public function getNextInvoiceId()
	{
		return get_next_id('invoices');
	}
	/**
	 * Generate recurring invoices
	 * @return type mixed
	 */
	public function generateInvoices()
	{
		$attachments = [];

		$this->load->model('invoices/invoices_model', 'invoices');
		$invoices = $this->invoices->getRecurrentInvoices();
		if ( ! empty( $invoices ) )
		{
			foreach ( $invoices as $key => $invoice )
			{

				$subtotal 					= $invoice->subtotal;
				$tax_amount 				= $invoice->tax_amount;
				$total 						= floatval($invoice->total);
				$amount_due 				= $invoice->amount_due;

				// Remove any forgotten late fees before duplicating invoice
				if ($invoice->late_fees_subtotal > 0) {
					$subtotal 					= $subtotal - $invoice->late_fees_subtotal;
					$tax_amount 				= $tax_amount - $invoice->late_fees_tax_amount;
					$total 						= $total - floatval($invoice->late_fees_subtotal + $invoice->late_fees_tax_amount);
					$amount_due 				= $total;
				}
				$this->client_id 			= $invoice->client_id;
				$user_id 			        = $invoice->user_id;
				$invoice_number 			= next_invoice_number(); // Get the next invoice number for insert
				$invoice_next_due_date 		= $invoice->next_duedate;
				$recurring_from 			= $invoice->next_duedate;
				$recurring_to 				= $invoice->recurring_to;
				$subject 					= $invoice->subject;
				$pay_type 					= $invoice->pay_type;
				
				
				
				if ($invoice->attachments) {
					$attachments = json_decode($invoice->attachments, true);
				}
				$discount_amount 			= $invoice->discount_amount;
				$amount_refunded 			= $invoice->amount_refunded;
				$notes 						= $invoice->notes;
				$date 						= $invoice_next_due_date;
				$default_duedate 			= get_option('default_duedate');
				$duedate 					= date( 'Y-m-d', strtotime( '+'.$default_duedate. 'day', strtotime( $date ) ) );
				// Get the change rate if multicurrency is enabled
				$currency = $this->client_id == 0 ? get_option('user_currency') : client( $this->client_id, 'client_currency' );
				$change_rate = _has_option( 'change_currency' ) ? get_change_rate( $currency ) : 1;

				switch ( $pay_type ) {
					case 2:
						$next_duedate = date( 'Y-m-d', strtotime( '+1 week', strtotime( $invoice_next_due_date ) ) );
						break;
					case 3:
						$next_duedate = date( 'Y-m-d', strtotime( '+2 week', strtotime( $invoice_next_due_date ) ) );
						break;
					case 4:
						$next_duedate = date( 'Y-m-d', strtotime( '+1 month', strtotime( $invoice_next_due_date ) ) );
						break;
					case 5:
						$next_duedate = date( 'Y-m-d', strtotime( '+1 year', strtotime( $invoice_next_due_date ) ) );
						break;
					case 6:
						$next_duedate = date( 'Y-m-d', strtotime( '+3 month', strtotime( $invoice_next_due_date ) ) );
						break;
					case 7:
						$next_duedate = date( 'Y-m-d', strtotime( '+6 month', strtotime( $invoice_next_due_date ) ) );
						break;
				}
				$invoice_array = array(
					'client_id' 			=> $this->client_id,
					'user_id' 				=> $user_id,
					'invoice_number' 		=> $invoice_number, // Be careful next invoice number value
					'subject' 				=> $subject,
					'pay_type' 				=> $pay_type,
					'date' 					=> $date,
					'due_date' 				=> $duedate,
					'recurring_from' 		=> $recurring_from,
					'recurring_to' 			=> $recurring_to,
					'next_duedate' 			=> $next_duedate,
					'subtotal' 				=> $subtotal,
					'tax_amount' 			=> $tax_amount,
					'discount_amount' 		=> $discount_amount,
					'total' 				=> $total,
					'amount_paid' 			=> 0,
					'amount_due' 			=> $total,
					'amount_refunded' 		=> $amount_refunded,
					'invoice_status' 		=> 'Unpaid',
					'notes' 				=> $notes,
					'sent' 					=> 'yes',
					'change_rate' 			=> $change_rate
				);
				// Set table and save data
				$this->invoices->set_table( 'invoices' );
				$this->invoices->save( $invoice_array );

				// Set Last Invoice frequency to once
				$db_data 		= array( 'pay_type' => 0 );
				$invoice_id 	= $invoice->id;

				# Update the last recurring invoice to exclude it from next cron
				$this->invoices->set_primary_key( 'id' );
				$this->invoices->save( $db_data, $invoice_id );

				// Fetch all recurrent items
				$items = $this->invoices->duplicate_invoiced_items( $invoice->invoice_number );
				if ($items) {
					foreach ($items as $key => $item) 
					{
						if (0 == $item->is_late_fees || $item->item_id != -1) {
							$item_array = array(
								'invoice_number' 			=> $invoice_number,
								'client_id' 				=> $item->client_id,
								'user_id' 					=> $item->user_id,
								'item_sku' 					=> $item->item_sku,
								'item_id' 					=> $item->item_id,
								'item_name' 				=> $item->item_name,
								'item_price' 				=> $item->item_price,
								'specific_serial_number' 	=> $item->specific_serial_number,
								'item_quantity' 			=> $item->item_quantity,
								'item_discount' 			=> $item->item_discount,
								'item_subtotal' 			=> $item->item_subtotal,
								'item_tax_rate' 			=> $item->item_tax_rate,
								'item_recurring' 			=> $item->item_recurring
							);
							$this->invoices->set_table('invoices_items');
							$this->invoices->save($item_array);
						}
						if ($this->client_id) {
							$data = array(
								'invoice_url'				=> base_url( 'index.php/viewinvoice/view?id=' ) . _eID( $invoice_number ),
								'to_name'            		=> html_entity_decode(client( $this->client_id, 'name_company' )),
								'date'		    			=> _fdate( client( $this->client_id, 'client_language' ), $date ),
								'due_date'		    		=> _fdate( client( $this->client_id, 'client_language' ), $duedate ),
								'invoice_status'			=> __( strtolower( $invoice_array['invoice_status'] ) ),
								'invoice_number' 			=> get_option( 'invoice_prefix' ).invoice_number_format() . sprintf( "%04s", $invoice_number ),
								'invoicesubject'			=> $subject
								);
							$data['invoice_amount'] 		= add_currency( format_number( $total * $change_rate ), $this->client_id );
							$data['amount_paid'] 			= add_currency( format_number( $invoice_array['amount_paid'] * $change_rate ), $this->client_id );
							$data['amount_due'] 			= add_currency( format_number( $amount_due * $change_rate ), $this->client_id );
							$data['amount_refunded'] 		= add_currency( format_number( $amount_refunded * $change_rate ), $this->client_id );
							$data['client_account'] 		= base_url( 'index.php/client/index?id=' ) . _eID( $this->client_id );
							$data['logo_url'] 				= base_url( 'uploads/admin/img/' ). get_option( 'logo' );
							$data['btn_client_account'] 	= __( 'btn_client_account', $this->client_id );

							$this->load->helper('file');

							$lang = lang_code_to_notif_string( client( $this->client_id, 'client_language' ) );
							$path_new_invoice = './notifications/' . $lang . '/invoices/new_invoice.php';
							$path_paid_invoice = './notifications/' . $lang . '/invoices/invoice_paid.php';
							$path_refunded_invoice = './notifications/' . $lang . '/invoices/invoice_refunded.php';

							$to = client( $this->client_id, 'client_email' );
							if ( $invoice_array['invoice_status'] === "Paid" )
							{
								$template = read_file( $path_paid_invoice );
								$subject = __( 'subject_payment_confirmation', $this->client_id );
							}
							elseif ( $invoice_array['invoice_status'] === "Refunded" )
							{
								$template = read_file( $path_refunded_invoice );
								$subject = __( 'subject_payment_refunded', $this->client_id );
							}
							else
							{
								$template = read_file( $path_new_invoice );
								$subject = __( 'subject_invoice_generated', $this->client_id );
							}
							$message = $this->parser->parse_string( $template, $data, true );

							if ($this->client_id) {
								$client_cc = json_decode( null !== client($this->client_id, 'client_cc' ) ? client($this->client_id, 'client_cc' ): json_encode( array() ));
								if (1 == get_option('attach_pdf_to_invoice')) {
									generate_pdf_file('invoices', $invoice_number);
								}
							}
							if (!_has_option('turn_off_customer_invoice_emails')) {
								$attachments[] = FCPATH.'uploads/pdf/invoices/'.$invoice_number.'.pdf';
								send_email($to, $message, $subject, null, $client_cc, $attachments, get_option('invoice_prefix').invoice_number_format() . sprintf("%04s", $invoice_number).'.pdf');
							}
						}
					}
				}
			}
		}
	}
	public function generateRecurringOrders()
	{
		$this->load->model('orders/orders_model', 'orders');
		$orders = $this->orders->getRecurrentOrders();
		if ( ! empty( $orders ) )
		{
			foreach ( $orders as $key => $order )
			{
				$this->client_id 			= $order->client_id;
				$user_id 			        = $order->user_id;
				$order_number 				= next_order_number(); // Get the next invoice number for insert
				$next_duedate 				= $order->next_duedate;
				$recurring_from 			= $order->recurring_from;
				$recurring_to 				= $order->recurring_to;
				$subject 					= $order->subject;
				$pay_type 					= $order->pay_type;
				$subtotal 					= $order->subtotal;
				$total 						= $order->total;
				$amount_due 				= $order->amount_due;
				$tax_amount 				= $order->tax_amount;
				$discount_amount 			= $order->discount_amount;
				$amount_refunded 			= $order->amount_refunded;
				$notes 						= $order->notes;
				$date 						= $next_duedate;
				$default_duedate 			= get_option('default_duedate');
				$duedate 					= date( 'Y-m-d', strtotime( '+'.$default_duedate. 'day', strtotime( $date ) ) );
				// Get the change rate if multicurrency is enabled
				$currency = $this->client_id == 0 ? get_option('user_currency') : client( $this->client_id, 'client_currency' );
				$change_rate = _has_option( 'change_currency' ) ? get_change_rate( $currency ) : 1;

				switch ( $pay_type ) {
					case 2:
						$next_duedate = date( 'Y-m-d', strtotime( '+1 week', strtotime( $next_duedate ) ) );
						break;
					case 3:
						$next_duedate = date( 'Y-m-d', strtotime( '+2 week', strtotime( $next_duedate ) ) );
						break;
					case 4:
						$next_duedate = date( 'Y-m-d', strtotime( '+1 month', strtotime( $next_duedate ) ) );
						break;
					case 5:
						$next_duedate = date( 'Y-m-d', strtotime( '+1 year', strtotime( $next_duedate ) ) );
						break;
					case 6:
						$next_duedate = date( 'Y-m-d', strtotime( '+3 month', strtotime( $next_duedate ) ) );
						break;
					case 7:
						$next_duedate = date( 'Y-m-d', strtotime( '+6 month', strtotime( $next_duedate ) ) );
						break;
				}
				$order_array = array(
					'client_id' 			=> $this->client_id,
					'user_id' 				=> $user_id,
					'order_number' 			=> $order_number, // Be careful next invoice number value
					'subject' 				=> $subject,
					'pay_type' 				=> $pay_type,
					'date' 					=> $date,
					'due_date' 				=> $duedate,
					'recurring_from' 		=> $recurring_from,
					'recurring_to' 			=> $recurring_to,
					'next_duedate' 			=> $next_duedate,
					'subtotal' 				=> $subtotal,
					'tax_amount' 			=> $tax_amount,
					'discount_amount' 		=> $discount_amount,
					'total' 				=> $total,
					'amount_paid' 			=> 0,
					'amount_due' 			=> $total,
					'amount_refunded' 		=> $amount_refunded,
					'order_status' 			=> 'Pending',
					'notes' 				=> $notes,
					'sent' 					=> 'yes',
					'change_rate' 			=> $change_rate
				);
				// Set table and save data
				$this->orders->set_table( 'orders' );
				$this->orders->save( $order_array );

				// Set Last order frequency to once
				$db_data 		= array( 'pay_type' => 0 );
				$order_id 		= $order->id;

				# Update the last recurring order to exclude it from next cron
				$this->orders->set_primary_key( 'id' );
				$this->orders->save( $db_data, $order_id );

				// Fetch all recurrent items
				$items = $this->orders->duplicate_ordered_items( $order->order_number );
				if ($items) {
					foreach ($items as $key => $item) 
					{
						$item_array = array(
							'order_number' 				=> $order_number,
							'client_id' 				=> $item->client_id,
							'user_id' 					=> $item->user_id,
							'item_sku' 					=> $item->item_sku,
							'item_id' 					=> $item->item_id,
							'item_name' 				=> $item->item_name,
							'item_price' 				=> $item->item_price,
							'item_quantity' 			=> $item->item_quantity,
							'item_discount' 			=> $item->item_discount,
							'item_subtotal' 			=> $item->item_subtotal,
							'specific_serial_number' 	=> $item->specific_serial_number,
							'item_tax_rate' 			=> $item->item_tax_rate,
							'item_recurring' 			=> $item->item_recurring
						);
						$this->orders->set_table('orders_items');
						$this->orders->save($item_array);

						if ($this->client_id) {
							$data = array(
								'order_url'					=> base_url( 'index.php/vieworder/view?id=' ) . _eID( $order_number ),
								'to_name'            		=> html_entity_decode(client( $this->client_id, 'name_company' )),
								'date'		    			=> _fdate( client( $this->client_id, 'client_language' ), $date ),
								'due_date'		    		=> _fdate( client( $this->client_id, 'client_language' ), $duedate ),
								'order_status'				=> __($order_array['order_status'] ),
								'order_number' 				=> get_option( 'order_prefix' ).invoice_number_format() . sprintf( "%04s", $order_number ),
								'ordersubject'				=> $subject
								);
							$data['order_amount'] 			= add_currency( format_number( $total * $change_rate ), $this->client_id );
							$data['amount_paid'] 			= add_currency( format_number( $order_array['amount_paid'] * $change_rate ), $this->client_id );
							$data['amount_due'] 			= add_currency( format_number( $amount_due * $change_rate ), $this->client_id );
							$data['amount_refunded'] 		= add_currency( format_number( $amount_refunded * $change_rate ), $this->client_id );
							$data['client_account'] 		= base_url( 'index.php/client/index?id=' ) . _eID( $this->client_id );
							$data['logo_url'] 				= base_url( 'uploads/admin/img/' ). get_option( 'logo' );
							$data['btn_client_account'] 	= __( 'btn_client_account', $this->client_id );

							$this->load->helper('file');
							$lang = lang_code_to_notif_string( client( $this->client_id, 'client_language' ) );
							$path_new_order = './notifications/' . $lang . '/orders/new_order.php';
							$path_shipped_order = './notifications/' . $lang . '/orders/order_shipped.php';

							$to = client( $this->client_id, 'client_email' );
							$template = ($order_array['order_status'] === "Shipped") ? read_file($path_shipped_order):
							read_file($path_new_order);
							$subject = ($order_array['order_status'] === "Shipped") ? __('subject_order_shipped_notification', $this->client_id) : __('subject_order_placed', $this->client_id);
							$message = ($order_array['order_status'] === "Shipped") ? 
							$this->parser->parse_string($template, $data, true) : 
							$this->parser->parse_string($template, $data, true);

							if ($this->client_id) {
								$client_cc = json_decode( null !== client($this->client_id, 'client_cc' ) ? client($this->client_id, 'client_cc' ): json_encode( array() ));
								if (1 == get_option('attach_pdf_to_invoice')) {
									generate_pdf_file('orders', $order_number);
								}
							}
							if (!_has_option('turn_off_customer_so_emails')) {
								send_email($to, $message, $subject, null, $client_cc, FCPATH.'uploads/pdf/orders/'.$order_number.'.pdf', get_option('order_prefix').invoice_number_format() . sprintf("%04s", $order_number).'.pdf');
							}
						}

						$event = array(
							'action' 					=> 'order_sent',
							'referrer' 					=> 'orders',
							'referrer_id' 				=> $order_number,
							'referrer_value' 			=> $order_number,
							'client_id' 				=> $this->client_id,
							'status' 					=> 'success'
						);
						record_event($event);
					}
				}
			}
		}
	}
	public function credit_note_invoice( $invoice_number )
	{
		$this->load->model( 'invoices/invoices_model', 'invoices' );
		$invoices = get_invoice_by_number( $invoice_number );
		if ( ! empty( $invoices ) )
		{
			foreach ( $invoices as $key => $invoice ) 
			{
				$client_id = $invoice->client_id;
				$user_id = $invoice->user_id;
				$key_number = next_invoice_number();

				// Get the change rate if multicurrency is enabled
				$currency = $invoice->client_id == 0 ? get_option('user_currency') : client( $invoice->client_id, 'client_currency' );
				$change_rate = _has_option( 'change_currency' ) ? get_change_rate( $currency ) : 1;

				$invoice_array 						= array(
					'client_id' 			=> $client_id,
					'user_id' 				=> $user_id,
					'invoice_number' 		=> next_invoice_number(),
					'invoice_key' 			=> $key_number,
					'subject' 				=> $invoice->subject,
					'pay_type' 				=> 1,
					'date' 					=> $invoice->date,
					'due_date' 				=> $invoice->due_date,
					'subtotal' 				=> $invoice->subtotal,
					'tax_amount' 			=> $invoice->tax_amount,
					'discount_amount' 		=> $invoice->discount_amount,
					'total'					=> $invoice->total,
					'amount_paid' 			=> 0,
					'amount_due' 			=> $invoice->total,
					'invoice_status' 		=> 'Refunded',
					'date_paid' 			=> $invoice->date_paid,
					'sent' 					=> 'no',
					'is_credit_note'		=> 1,
					'initial_invoice_number' => $invoice_number,
					'change_rate' 			=> $change_rate
				);
				// Set table and save data
				$this->invoices->set_table( 'invoices' );
				$this->invoices->save( $invoice_array );

				// Update next invoice ID
				setAccountEntityId('invoice', $key_number);

				// Fetch all recurrent items
				$items = get_invoiced_items( $invoice_number );

				foreach ( $items as $key => $item ) 
				{
					$item_array = array(
						'invoice_number' 	=> $invoice_array['invoice_number'],
						'client_id' 		=> $item->client_id,
						'user_id' 			=> $item->user_id,
						'item_sku' 			=> $item->item_sku,
						'item_id' 			=> $item->item_id,
						'item_name' 		=> $item->item_name,
						'item_price' 		=> $item->item_price,
						'item_quantity'		=> $item->item_quantity,
						'item_discount' 	=> $item->item_discount,
						'specific_serial_number' 		=> $item->specific_serial_number,
						'item_subtotal' 	=> $item->item_subtotal,
						'item_tax_rate' 	=> $item->item_tax_rate,
					);
					$this->invoices->set_table( 'invoices_items' );
					$this->invoices->save( $item_array );
				}
			}
		}
	}

	public function apply_late_fees( $invoice_number )
	{
		$this->load->model( 'invoices/invoices_model', 'invoices' );
		$invoices = get_invoice_by_number( $invoice_number );
		$late_fees_subtotal = 0;

		if ( ! empty( $invoices ) )
		{
			foreach ( $invoices as $key => $invoice ) 
			{
				try {
					// Let's see if there is any late fees amount set in invoice table
					$late_fees_subtotal = $invoice->late_fees_subtotal;

					if (null == $late_fees_subtotal || 0 == $late_fees_subtotal) {
						// Let's see if there is any late fees amount set in invoice items table
						try {
							$late_fees_subtotal = _has_late_fees($invoice_number);
							// Let's apply late fees
							if(null == $late_fees_subtotal || 0 == $late_fees_subtotal) {
								$client_id = $invoice->client_id;
								$user_id = $invoice->user_id;
								// Client is not excluded from late fees
								if ( ! client($client_id, 'exclude_late_fees')) {

									$this->set_invoice_late_fees($invoice_number, $client_id, $user_id);
								}
							}
						} catch (Exception $e) {
							//
						}
					}
				} catch (Exception $e) {
					//
				}
			}
		}
	}

	public function set_invoice_late_fees($invoice_number, $client_id, $user_id) {

		$this->load->model( 'invoices/invoices_model', 'invoices' );
		$late_fees_value = get_option('late_fees_value'); // Set in oc_options, i.e $10 or 10% of amount, so defined value, use to compute
		$fees_type = get_option('late_fees_type'); // fixed or rate

		// Let's define item array
		$item_price = 0;

		// get the invoice
		$invoice = get_invoice_by_number( $invoice_number );

		// Does vat apply on late fees ?
		if (null != get_option('late_fees_tax_rate') && get_option('late_fees_tax_rate') > 0) {
			// fixed amount
			if ($fees_type == "fixed") {
				// late fees unit price in invoice template
				$item_price = $late_fees_value;
				$item_line_subtotal = $item_price;
				$item_tax_amount = ($item_line_subtotal * get_option('late_fees_tax_rate'))/100;
			} else {
				// So late fees unit price in invoice template is computed base on invoice subtotal
				$item_price = ($invoice[0]->subtotal * get_option('late_fees_value')/100);
				$item_line_subtotal = $item_price;
				$item_tax_amount = ($item_line_subtotal * get_option('late_fees_tax_rate'))/100;
			}
		} else {
			if ($fees_type == "fixed") {
				$item_price = $late_fees_value;
				$item_line_subtotal = $item_price;
				$item_tax_amount = 0;
			} else {
				$item_price = ($invoice[0]->subtotal * get_option('late_fees_value')/100);
				$item_line_subtotal = $item_price;
				$item_tax_amount = 0;
			}
		}
		
		// Save late_fees to invoice table
		$subtotal = $invoice[0]->subtotal + $item_line_subtotal;
		$tax_amount = $invoice[0]->tax_amount + $item_tax_amount;
		$total = $subtotal + $tax_amount;
		$invoice_array 				= array(
			'late_fees_subtotal' 	    => $item_line_subtotal,
			'late_fees_tax_amount' 	    => $item_tax_amount,
			'late_fees_tax_rate' 	    => get_option('late_fees_tax_rate'),
			'subtotal' 				    => $subtotal,
			'tax_amount' 			    => $tax_amount,
			'total' 				    => $total,
			'amount_paid' 			    => $invoice[0]->amount_paid,
			'amount_due' 			    => $total - $invoice[0]->amount_paid
		);
		// Set table and save data
		$this->invoices->set_table( 'invoices' );
		$this->invoices->save( $invoice_array, $invoice_number );

		// Apply late fees
		$item_array = array(
			'invoice_number' 	=> $invoice_number,
			'client_id' 		=> $client_id,
			'user_id' 			=> $user_id,
			'item_sku' 			=> 'LF-99',
			'item_id' 			=> '-1',
			'item_name' 		=> __('label_late_fees'),
			'item_price' 		=> $item_price,
			'item_quantity'		=> 1,
			'item_discount' 	=> 0,
			'specific_serial_number' 		=> '',
			'item_subtotal' 	=> $item_line_subtotal,
			'item_tax_rate' 	=> get_option('late_fees_tax_rate'),
			'is_late_fees' 	    => 1,
			);

		$this->invoices->set_table('invoices_items');
		$this->invoices->save($item_array);
	}
	/**
	 * Send Invoice Notification email
	 * @param type $invoice_number 
	 * @return type mixed
	 */
	public function send_invoice_email( $invoice_number )
	{
		$attachments = [];

		$invoices = get_invoice_by_number($invoice_number);
		if ( !empty($invoices) )
		{
			foreach ($invoices as $key => $invoice) 
			{
				$client_language 			= client($invoice->client_id, 'client_language');
				$client_id 					= $invoice->client_id;
				$invoice_number 			= $invoice->invoice_number;
				$subject 					= $invoice->subject;
				$pay_type 					= $invoice->pay_type;
				$date 						= $invoice->date;
				$due_date 					= $invoice->due_date;
				$subtotal 					= $invoice->subtotal;
				$transaction_id 			= isset($invoice->transaction_id) ? $invoice->transaction_id : time();
				$tax_amount 				= $invoice->tax_amount;
				$discount_amount 			= $invoice->discount_amount;
				$total 						= $invoice->total;
				$amount_paid 				= $invoice->amount_paid;
				$amount_due 				= $invoice->amount_due;
				$amount_refunded 			= $invoice->amount_refunded;
				$invoice_status 			= $invoice->invoice_status;
				$notes 						= $invoice->notes;
				$sent 						= 'yes';

				if ($invoice->attachments) {
					$attachments = json_decode($invoice->attachments, true);
				}
				// Get the change rate if multicurrency is enabled
				$currency 					= $client_id == 0 ? get_option('user_currency') : client($client_id, 'client_currency');
				$change_rate 				= _has_option('change_currency') ? get_change_rate( $currency ) : 1;

				$invoice_array = array(
					'client_id' 			=> $client_id,
					'invoice_number' 		=> $invoice_number,
					'subject' 				=> $subject,
					'pay_type' 				=> $pay_type,
					'date' 					=> $date,
					'due_date' 				=> $due_date,
					'subtotal' 				=> $subtotal,
					'transaction_id' 		=> $transaction_id,
					'tax_amount' 			=> $tax_amount,
					'discount_amount' 		=> $discount_amount,
					'total' 				=> $total,
					'amount_paid' 			=> $amount_paid,
					'amount_due' 			=> $amount_due,
					'amount_refunded' 		=> $amount_refunded,
					'invoice_status' 		=> $invoice_status,
					'notes' 				=> $notes,
					'sent' 					=> $sent,
					'change_rate' 			=> $change_rate
				);
				if ($client_id) {
					$data = array(
						'invoice_url'			=> base_url( 'index.php/viewinvoice/view?id=' ) . _eID( $invoice_number ),
						'to_name'            	=> html_entity_decode(client( $client_id, 'name_company' )),
						'date'		    		=> _fdate( $client_language, $date ),
						'due_date'		    	=> _fdate( $client_language, $due_date ),
						'invoice_status'		=> __( strtolower( $invoice_status ), $client_id ),
						'invoice_number' 		=> get_option('invoice_prefix').invoice_number_format() . sprintf("%04s", $invoice_number),
						'invoicesubject'		=> $subject
						);

					$data['invoice_amount'] 	= add_currency( format_number( $total * $change_rate ), $client_id );
					$data['amount_paid'] 		= add_currency( format_number( $amount_paid * $change_rate ), $client_id );
					$data['transaction_id'] 	= $transaction_id;
					$data['amount_due'] 		= add_currency( format_number( $amount_due * $change_rate ), $client_id );
					$data['amount_refunded'] 	= add_currency( format_number( $amount_refunded * $change_rate ), $client_id );
					$data['client_account'] 	= base_url( 'index.php/client/index?id=' ) . _eID( $client_id );
					$data['logo_url'] 			= base_url( 'uploads/admin/img/' ). get_option( 'logo' );
					$data['btn_client_account'] = __( 'btn_client_account', $client_id );

					$this->load->helper('file');
					$lang 						= lang_code_to_notif_string( client( $client_id, 'client_language' ) );
					$path_new_invoice 			= './notifications/' . $lang . '/invoices/new_invoice.php';
					$path_paid_invoice 			= './notifications/' . $lang . '/invoices/invoice_paid.php';
					$path_refunded_invoice 		= './notifications/' . $lang . '/invoices/invoice_refunded.php';

					$to 	= client( $client_id, 'client_email' );
					if ( $invoice_status === "Paid" && ! $amount_refunded )
					{
						$template 	= read_file( $path_paid_invoice );
						$subject 	= __( 'subject_payment_confirmation', $client_id );
					}
					elseif( $invoice_status === "Refunded" || $amount_refunded > 0 )
					{
						$template 	= read_file( $path_refunded_invoice );
						$subject 	= __( 'subject_payment_refunded', $client_id );
					}
					else
					{
						$template 	= read_file( $path_new_invoice );
						$subject 	= __( 'subject_invoice_generated', $client_id );
					}
					$message 	= $this->parser->parse_string($template, $data, true);

					if ($client_id) {
						$client_cc = json_decode( null !== client($client_id, 'client_cc' ) ? client($client_id, 'client_cc' ): json_encode( array() ));
						if (1 == get_option('attach_pdf_to_invoice')) {
							generate_pdf_file('invoices', $invoice_number);
						}
					}
					if ( !_has_option('turn_off_customer_invoice_emails')) {
						$attachments[] = FCPATH.'uploads/pdf/invoices/'.$invoice_number.'.pdf';
						send_email($to, $message, $subject, null, $client_cc, $attachments, get_option('invoice_prefix').invoice_number_format() . sprintf("%04s", $invoice_number).'.pdf');
					}
				}
			}
		}
		return false;
	}
	public function send_proforma_email( $proforma_number )
	{
		$attachments = [];
		$invoices = get_proforma_by_number($proforma_number);
		if ( !empty($invoices) )
		{
			foreach ($invoices as $key => $invoice) 
			{
				$client_language 			= client($invoice->client_id, 'client_language');
				$client_id 					= $invoice->client_id;
				$proforma_number 			= $invoice->proforma_number;
				$subject 					= $invoice->subject;
				$pay_type 					= 1;
				$date 						= $invoice->date;
				$due_date 					= $invoice->due_date;
				$subtotal 					= $invoice->subtotal;
				$tax_amount 				= $invoice->tax_amount;
				$discount_amount 			= $invoice->discount_amount;
				$total 						= $invoice->total;
				$amount_paid 				= $invoice->amount_paid;
				$amount_due 				= $invoice->amount_due;
				$amount_refunded 			= $invoice->amount_refunded;
				$proforma_status 			= $invoice->proforma_status;
				$notes 						= $invoice->notes;
				$sent 						= 'yes';

				if ($invoice->attachments) {
					$attachments = json_decode($invoice->attachments, true);
				}
				// Get the change rate if multicurrency is enabled
				$currency 					= $client_id == 0 ? get_option('user_currency') : client($client_id, 'client_currency');
				$change_rate 				= _has_option('change_currency') ? get_change_rate( $currency ) : 1;

				$invoice_array = array(
					'client_id' 			=> $client_id,
					'invoice_number' 		=> $proforma_number,
					'subject' 				=> $subject,
					'pay_type' 				=> $pay_type,
					'date' 					=> $date,
					'due_date' 				=> $due_date,
					'subtotal' 				=> $subtotal,
					'tax_amount' 			=> $tax_amount,
					'discount_amount' 		=> $discount_amount,
					'total' 				=> $total,
					'amount_paid' 			=> $amount_paid,
					'amount_due' 			=> $amount_due,
					'amount_refunded' 		=> $amount_refunded,
					'proforma_status' 		=> $proforma_status,
					'notes' 				=> $notes,
					'sent' 					=> $sent,
					'change_rate' 			=> $change_rate
				);
				if ($client_id) {
					$data = array(
						'proforma_url'			=> base_url( 'index.php/viewproforma/view?id=' ) . _eID( $proforma_number ),
						'to_name'            	=> html_entity_decode(client( $client_id, 'name_company' )),
						'date'		    		=> _fdate( $client_language, $date ),
						'due_date'		    	=> _fdate( $client_language, $due_date ),
						'proforma_status'		=> __( strtolower( $proforma_status ), $client_id ),
						'proforma_number' 		=> get_option('proforma_prefix').invoice_number_format() . sprintf("%04s", $proforma_number),
						'invoicesubject'		=> $subject
						);

					$data['proforma_amount'] 	= add_currency( format_number( $total * $change_rate ), $client_id );
					$data['amount_paid'] 		= add_currency( format_number( $amount_paid * $change_rate ), $client_id );
					$data['amount_due'] 		= add_currency( format_number( $amount_due * $change_rate ), $client_id );
					$data['amount_refunded'] 	= add_currency( format_number( $amount_refunded * $change_rate ), $client_id );
					$data['client_account'] 	= base_url( 'index.php/client/index?id=' ) . _eID( $client_id );
					$data['logo_url'] 			= base_url( 'uploads/admin/img/' ). get_option( 'logo' );
					$data['btn_client_account'] = __( 'btn_client_account', $client_id );

					$this->load->helper('file');
					$lang 						= lang_code_to_notif_string( client( $client_id, 'client_language' ) );
					$path_new_invoice 			= './notifications/' . $lang . '/proforma/new_proforma.php';

					$to 	= client( $client_id, 'client_email' );
					
					$template 	= read_file( $path_new_invoice );
					$subject 	= __( 'subject_proforma_generated', $client_id );
					$message 	= $this->parser->parse_string($template, $data, true);

					if ($client_id) {
						$client_cc = json_decode( null !== client($client_id, 'client_cc' ) ? client($client_id, 'client_cc' ): json_encode( array() ));
						if (1 == get_option('attach_pdf_to_invoice')) {
							generate_pdf_file('proforma', $proforma_number);
						}
					}

					if ( !_has_option('turn_off_customer_proforma_emails')) {
						$attachments[] = FCPATH.'uploads/pdf/proforma/'.$proforma_number.'.pdf';
						send_email($to, $message, $subject, null, $client_cc, $attachments, get_option('proforma_prefix').invoice_number_format() . sprintf("%04s", $proforma_number).'.pdf');
					}
				}
			}
		}
		return false;
	}
	public function send_order_email( $order_number )
	{
		$orders = get_order_by_number($order_number);
		if ( !empty($orders) )
		{
			foreach ($orders as $key => $order) 
			{
				$client_language 			= client($order->client_id, 'client_language');
				$client_id 					= $order->client_id;
				$order_number 				= $order->order_number;
				$subject 					= $order->subject;
				$pay_type 					= $order->pay_type;
				$date 						= $order->date;
				$due_date 					= $order->due_date;
				$subtotal 					= $order->subtotal;
				$transaction_id 			= isset($order->transaction_id) ? $order->transaction_id : time();
				$tax_amount 				= $order->tax_amount;
				$discount_amount 			= $order->discount_amount;
				$total 						= $order->total;
				$amount_paid 				= $order->amount_paid;
				$amount_due 				= $order->amount_due;
				$amount_refunded 			= $order->amount_refunded;
				$order_status 				= $order->order_status;
				$notes 						= $order->notes;
				$sent 						= 'yes';
				// Get the change rate if multicurrency is enabled
				$currency 					= $client_id == 0 ? get_option('user_currency') : client($client_id, 'client_currency');
				$change_rate 				= _has_option('change_currency') ? get_change_rate( $currency ) : 1;

				$order_array = array(
					'client_id' 			=> $client_id,
					'order_number' 			=> $order_number,
					'subject' 				=> $subject,
					'pay_type' 				=> $pay_type,
					'date' 					=> $date,
					'due_date' 				=> $due_date,
					'subtotal' 				=> $subtotal,
					'transaction_id' 		=> $transaction_id,
					'tax_amount' 			=> $tax_amount,
					'discount_amount' 		=> $discount_amount,
					'total' 				=> $total,
					'amount_paid' 			=> $amount_paid,
					'amount_due' 			=> $amount_due,
					'amount_refunded' 		=> $amount_refunded,
					'order_status' 			=> $order_status,
					'notes' 				=> $notes,
					'sent' 					=> $sent,
					'change_rate' 			=> $change_rate
				);
				if ($client_id) {
					$data = array(
						'order_url'				=> base_url( 'index.php/vieworder/view?id=' ) . _eID( $order_number ),
						'to_name'            	=> html_entity_decode(client( $client_id, 'name_company' )),
						'date'		    		=> _fdate( $client_language, $date ),
						'due_date'		    	=> _fdate( $client_language, $due_date ),
						'order_status'			=> __( strtolower( $order_status ), $client_id ),
						'order_number' 			=> get_option('order_prefix').invoice_number_format() . sprintf("%04s", $order_number),
						'invoicesubject'		=> $subject
						);

					$data['order_amount'] 		= add_currency( format_number( $total * $change_rate ), $client_id );
					$data['amount_paid'] 		= add_currency( format_number( $amount_paid * $change_rate ), $client_id );
					$data['transaction_id'] 	= $transaction_id;
					$data['amount_due'] 		= add_currency( format_number( $amount_due * $change_rate ), $client_id );
					$data['amount_refunded'] 	= add_currency( format_number( $amount_refunded * $change_rate ), $client_id );
					$data['client_account'] 	= base_url( 'index.php/client/index?id=' ) . _eID( $client_id );
					$data['logo_url'] 			= base_url( 'uploads/admin/img/' ). get_option( 'logo' );
					$data['btn_client_account'] = __( 'btn_client_account', $client_id );

					$this->load->helper('file');
					$lang 						= lang_code_to_notif_string( client( $client_id, 'client_language' ) );
					$path_new_order = './notifications/' . $lang . '/orders/new_order.php';
					$path_shipped_order = './notifications/' . $lang . '/orders/order_shipped.php';

					$to = client( $client_id, 'client_email' );
					$template = ($order_status === "Shipped") ? read_file($path_shipped_order):
					read_file($path_new_order);
					$subject = ($order_status === "Shipped") ? __('subject_order_shipped_notification', $client_id) : __('subject_order_placed', $client_id);
					$message = ($order_status === "Shipped") ? 
					$this->parser->parse_string($template, $data, true) : 
					$this->parser->parse_string($template, $data, true);

					if ($client_id) {
						$client_cc = json_decode( null !== client($client_id, 'client_cc' ) ? client($client_id, 'client_cc' ): json_encode( array() ));
						if (1 == get_option('attach_pdf_to_invoice')) {
							generate_pdf_file('orders', $order_number);
						}
					}

					if ( !_has_option('turn_off_customer_so_emails')) {
						send_email($to, $message, $subject, null, $client_cc, FCPATH.'uploads/pdf/orders/'.$order_number.'.pdf', get_option('order_prefix').invoice_number_format() . sprintf("%04s", $order_number).'.pdf');
					}
				}

				$event = array(
					'action' 					=> 'order_sent',
					'referrer' 					=> 'orders',
					'referrer_id' 				=> $order_number,
					'referrer_value' 			=> $order_number,
					'client_id' 				=> $client_id,
					'status' 					=> 'success'
				);
				record_event($event);
			}
		}
		return false;
	}
	public function send_purchase_email( $purchase_number )
	{
		$attachments = [];
		$invoices = get_purchase_by_number($purchase_number);
		if ( !empty($invoices) )
		{
			foreach ($invoices as $key => $invoice) 
			{
				$provider_language 			= provider($invoice->provider_id, 'provider_language');
				$provider_id 				= $invoice->provider_id;
				$purchase_number 			= $invoice->purchase_number;
				$subject 					= $invoice->subject;
				$pay_type 					= $invoice->pay_type;
				$date 						= $invoice->date;
				$shipping_date 				= $invoice->shipping_date;
				$subtotal 					= $invoice->subtotal;
				$payment_id 				= isset($invoice->payment_id) ? $invoice->payment_id : time();
				$tax_amount 				= $invoice->tax_amount;
				$discount_amount 			= $invoice->discount_amount;
				$total 						= $invoice->total;
				$amount_paid 				= $invoice->amount_paid;
				$amount_due 				= $invoice->amount_due;
				$amount_refunded 			= $invoice->amount_refunded;
				$purchase_status 			= $invoice->purchase_status;
				$notes 						= $invoice->notes;
				$sent 						= 'yes';
				$attachments 				= json_decode($invoice->attachments, true);
				// Get the change rate if multicurrency is enabled
				$currency 					= $provider_id == 0 ? get_option('user_currency') : provider($provider_id, 'provider_currency');
				$change_rate 				= _has_option('change_currency') ? get_change_rate( $currency ) : 1;

				$invoice_array = array(
					'provider_id' 					=> $provider_id,
					'purchase_number' 				=> $purchase_number,
					'subject' 						=> $subject,
					'pay_type' 						=> $pay_type,
					'date' 							=> $date,
					'shipping_date' 				=> $shipping_date,
					'subtotal' 						=> $subtotal,
					'payment_id' 					=> $payment_id,
					'tax_amount' 					=> $tax_amount,
					'discount_amount' 				=> $discount_amount,
					'total' 						=> $total,
					'amount_paid' 					=> $amount_paid,
					'amount_due' 					=> $amount_due,
					'amount_refunded' 				=> $amount_refunded,
					'purchase_status' 		    	=> $purchase_status,
					'notes' 						=> $notes,
					'sent' 							=> $sent,
					'change_rate' 					=> $change_rate
				);
				if ($provider_id) {
					$data = array(
						'purchase_url'			=> base_url( 'index.php/viewpurchase/view?id=' ) . _eID( $purchase_number ),
						'to_name'            	=> html_entity_decode(provider( $provider_id, 'name_company' )),
						'date'		    		=> _fdate( $provider_language, $date ),
						'shipping_date'		    => _fdate( $provider_language, $shipping_date ),
						'purchase_status'			=> __( strtolower( $purchase_status ), $provider_id ),
						'purchase_number' 		=> get_option('purchase_order_prefix').invoice_number_format() . sprintf("%04s", $purchase_number),
						'invoicesubject'		=> $subject
						);

					$data['purchase_amount'] 		= add_provider_currency( format_number( $total * $change_rate ), $provider_id );
					$data['amount_paid'] 		= add_provider_currency( format_number( $amount_paid * $change_rate ), $provider_id );
					$data['transaction_id'] 	= $payment_id;
					$data['amount_due'] 		= add_provider_currency( format_number( $amount_due * $change_rate ), $provider_id );
					$data['amount_refunded'] 	= add_provider_currency( format_number( $amount_refunded * $change_rate ), $provider_id );
					$data['provider_account'] 	= base_url( 'index.php/provider/index?id=' ) . _eID( $provider_id );
					$data['logo_url'] 			= base_url( 'uploads/admin/img/' ). get_option( 'logo' );
					$data['btn_provider_account'] = __( 'btn_provider_account', $provider_id );

					$this->load->helper('file');
					$lang 						= lang_code_to_notif_string( provider( $provider_id, 'provider_language' ) );
					$path_new_purchase 			= './notifications/' . $lang . '/purchases/new_purchase.php';
					$path_purchase_paid 			= './notifications/' . $lang . '/purchases/purchase_paid.php';

					if ( $purchase_status === "Paid" )
					{
						$template 	= read_file( $path_purchase_paid );
						$subject 	= __( 'subject_paid_purchase', $provider_id );
					}
					else
					{
						$template 					= read_file( $path_new_purchase );
						$subject 					= __( 'subject_purchase_order', $provider_id );
					}
					
					$to 						= provider( $provider_id, 'provider_email' );
					$message 					= $this->parser->parse_string($template, $data, true);

					if ( !_has_option('turn_off_customer_po_emails')) {
						send_email($to, $message, $subject, null, null, $attachments);
					}
				}
			}
		}
		return false;
	}
	/**
	 * Convert accepted quote to invoice
	 * @param type $quote_number 
	 * @return type mixed
	 */
	public function from_quote_to_invoice( $quote_number, $context=null)
	{
		$this->load->model( 'invoices/invoices_model', 'invoices' );
		$attachments = [];

		$invoices = get_quote_by_number( $quote_number );
		if ( ! empty( $invoices ) )
		{
			foreach ( $invoices as $key => $invoice ) 
			{
				$client_id = $invoice->client_id;
				// Get the change rate if multicurrency is enabled
				$currency = $invoice->client_id == 0 ? get_option('user_currency') : client( $invoice->client_id, 'client_currency' );
				$change_rate = _has_option( 'change_currency' ) ? get_change_rate( $currency ) : 1;
				$invoice_number = next_invoice_number();
				// Current date
				$date = date( "Y-m-d H:i:s" );

				// Default due
				$default_due_date 					= get_option( 'default_duedate' );
				$due_date 							= date( 'Y-m-d', strtotime( '+'.$default_due_date. 'day', strtotime( $date ) ) );
				$invoice_array 						= array(
					'client_id' 			=> $invoice->client_id,
					'user_id' 				=> $invoice->user_id,
					'invoice_number' 		=> $invoice_number,
					'subject' 				=> $invoice->subject,
					'pay_type' 				=> 1,
					'date' 					=> $date,
					'due_date' 				=> $due_date,
					'subtotal' 				=> $invoice->subtotal,
					'tax_amount' 			=> $invoice->tax_amount,
					'discount_amount' 		=> $invoice->discount_amount,
					'total'					=> $invoice->total,
					'amount_paid' 			=> 0,
					'amount_due' 			=> $invoice->total,
					'invoice_status' 		=> 'Unpaid',
					'sent' 					=> 'yes',
					'attachments' 			=> $invoice->attachments,
					'change_rate' 			=> $change_rate
				);
				if (has_repair_module()) {
					$invoice_array['repair_number'] = $invoice->repair_number;
				}

				if ($invoice->attachments){
					$attachments = json_decode($invoice->attachments, true);
				}
				// Set table and save data
				$this->invoices->set_table( 'invoices' );
				$this->invoices->save( $invoice_array );

				//$invoice_id = $invoice->id;

				if (has_repair_module()) {
					if ($context === 'repair') {
						set_repair_vars($invoice->repair_number, array('invoice_number' => $invoice_number));
					}
				}
				
				// Fetch all recurrent items
				$items = import_quoted_items( $quote_number );

				foreach ( $items as $key => $item ) 
				{
					$item_array = array(
						'invoice_number' 	=> $invoice_number,
						'client_id' 		=> $item->client_id,
						'user_id' 			=> $item->user_id,
						'item_sku' 			=> $item->item_sku,
						'item_id' 			=> $item->item_id,
						'specific_serial_number' 		=> $item->specific_serial_number,
						'item_name' 		=> $item->item_name,
						'item_price' 		=> $item->item_price,
						'item_quantity'		=> $item->item_quantity,
						'item_discount' 	=> $item->item_discount,
						'item_subtotal' 	=> $item->item_subtotal,
						'item_tax_rate' 	=> $item->item_tax_rate,
					);

					if (has_repair_module()) {
						$item_array['repair_number'] = $item->repair_number;
					}
					$this->invoices->set_table( 'invoices_items' );
					$this->invoices->save( $item_array );
				}

				if ($client_id) {
					$data = array(
						'invoice_url'		=> base_url( 'index.php/viewinvoice/view?id=' ) . _eID( $invoice_array['invoice_number'] ),
						'to_name'           => html_entity_decode(client( $invoice_array['client_id'], 'name_company' )),
						'date'		    	=> _fdate( client( $client_id, 'client_language' ), $invoice_array['date'] ),
						'due_date'		    => _fdate( client( $client_id, 'client_language' ), $invoice_array['due_date'] ),
						'invoice_status'	=> __( strtolower( $invoice_array['invoice_status'] ), $invoice_array['client_id'] ),
						'invoice_number' 	=> get_option( 'invoice_prefix' ).invoice_number_format() . sprintf( "%04s", $invoice_array['invoice_number'] ),
						'invoicesubject'	=> $invoice_array['subject']
						);

					$data['invoice_amount'] 	= add_currency( format_number( $invoice_array['total'] * $invoice_array['change_rate'] ), $invoice_array['client_id'] );
					$data['amount_paid'] 		= add_currency( format_number( $invoice_array['amount_paid'] * $invoice_array['change_rate'] ), $invoice_array['client_id'] );
					$data['amount_due'] 		= add_currency(format_number($invoice_array['amount_due'] * $invoice_array['change_rate']), $invoice_array['client_id']);
					$data['client_account'] 	= base_url( 'index.php/client/index?id=' ) . _eID( $invoice_array['client_id'] );
					$data['logo_url'] 			= base_url( 'uploads/admin/img/' ). get_option( 'logo' );
					$data['btn_client_account'] = __( 'btn_client_account', $invoice_array['client_id'] );

					$this->load->helper('file');
					$lang 					= lang_code_to_notif_string( client( $client_id, 'client_language' ) );
					$path_new_invoice 		= './notifications/' . $lang . '/invoices/new_invoice.php';
					$path_paid_invoice 		= './notifications/' . $lang . '/invoices/invoice_paid.php';
					$path_refunded_invoice 	= './notifications/' . $lang . '/invoices/invoice_refunded.php';

					$to = client( $invoice_array['client_id'], 'client_email' );
					
					$template = ( $invoice_array['invoice_status'] === "Paid" ) ? read_file( $path_paid_invoice ):
					read_file( $path_new_invoice );
					$subject = ( $invoice_array['invoice_status'] === "Paid" ) ? __( 'subject_payment_confirmation', $invoice_array['client_id'] ) : __( 'subject_invoice_generated', $invoice_array['client_id'] );
					$message = ( $invoice_array['invoice_status'] === "Paid" ) ? 
					$this->parser->parse_string( $template, $data, true ) : 
					$this->parser->parse_string( $template, $data, true );

					if ($client_id) {
						$client_cc = json_decode( null !== client($client_id, 'client_cc' ) ? client($client_id, 'client_cc' ): json_encode( array() ));
						if (1 == get_option('attach_pdf_to_invoice')) {
							generate_pdf_file('invoices', $invoice_number);
						}
					}
					if ( !_has_option('turn_off_customer_invoice_emails')) {
						$attachments[] = FCPATH.'uploads/pdf/invoices/'.$invoice_number.'.pdf';
						send_email($to, $message, $subject, null, $client_cc, $attachments, get_option('invoice_prefix').invoice_number_format() . sprintf("%04s", $invoice_number).'.pdf');
					}
				}
			}
		}
		return false;
	}
	public function from_order_to_invoice( $order_number )
	{
		$this->load->model( 'invoices/invoices_model', 'invoices' );
		$invoices = get_order_by_number( $order_number );
		if ( ! empty( $invoices ) )
		{
			foreach ( $invoices as $key => $invoice ) 
			{
				$client_id = $invoice->client_id;
				// Get the change rate if multicurrency is enabled
				$currency = $invoice->client_id == 0 ? get_option('user_currency') : client( $invoice->client_id, 'client_currency' );
				$change_rate = _has_option( 'change_currency' ) ? get_change_rate( $currency ) : 1;
				$invoice_number = next_invoice_number();
				// Current date
				$date = date( "Y-m-d H:i:s" );

				// Default due
				$default_due_date 					= get_option( 'default_duedate' );
				$due_date 							= date( 'Y-m-d', strtotime( '+'.$default_due_date. 'day', strtotime( $date ) ) );
				$invoice_array 						= array(
					'client_id' 			=> $invoice->client_id,
					'user_id' 				=> $invoice->user_id,
					'invoice_number' 		=> $invoice_number,
					'order_number' 			=> $order_number,
					'invoice_key' 			=> $invoice_number,
					'subject' 				=> $invoice->subject,
					'pay_type' 				=> 1,
					'date' 					=> $date,
					'due_date' 				=> $due_date,
					'subtotal' 				=> $invoice->subtotal,
					'tax_amount' 			=> $invoice->tax_amount,
					'discount_amount' 		=> $invoice->discount_amount,
					'total'					=> $invoice->total,
					'amount_paid' 			=> 0,
					'amount_due' 			=> $invoice->total,
					'invoice_status' 		=> 'Unpaid',
					'sent' 					=> 'yes',
					'change_rate' 			=> $change_rate
				);
				// Set table and save data
				$this->invoices->set_table( 'invoices' );
				$this->invoices->save( $invoice_array );

				//$invoice_id = $invoice->id;

				// Fetch all recurrent items
				$items = get_sale_items( $order_number );

				foreach ( $items as $key => $item ) 
				{
					$item_array = array(
						'invoice_number' 	=> $invoice_number,
						'invoice_key' 		=> $invoice_number,
						'client_id' 		=> $item->client_id,
						'user_id' 			=> $item->user_id,
						'item_sku' 			=> $item->item_sku,
						'item_id' 			=> $item->item_id,
						'item_group' 		=> $this->m_manager->category_id($item->item_id),
						'item_name' 		=> $item->item_name,
						'specific_serial_number' 		=> $item->specific_serial_number,
						'item_price' 		=> $item->item_price,
						'item_quantity'		=> $item->item_quantity,
						'item_discount' 	=> $item->item_discount,
						'item_subtotal' 	=> $item->item_subtotal,
						'item_tax_rate' 	=> $item->item_tax_rate,
						'item_uom' 			=> $item->item_uom
					);
					$this->invoices->set_table( 'invoices_items' );
					$this->invoices->save( $item_array );
					remove_committed_items($invoice_number);
				}
				if ($client_id) {
					$data = array(
						'invoice_url'		=> base_url( 'index.php/viewinvoice/view?id=' ) . _eID( $invoice_array['invoice_number'] ),
						'to_name'           => html_entity_decode(client( $invoice_array['client_id'], 'name_company' )),
						'date'		    	=> _fdate( client( $client_id, 'client_language' ), $invoice_array['date'] ),
						'due_date'		    => _fdate( client( $client_id, 'client_language' ), $invoice_array['due_date'] ),
						'invoice_status'	=> __( strtolower( $invoice_array['invoice_status'] ), $invoice_array['client_id'] ),
						'invoice_number' 	=> get_option( 'invoice_prefix' ).invoice_number_format() . sprintf( "%04s", $invoice_array['invoice_number'] ),
						'invoicesubject'	=> $invoice_array['subject']
						);

					$data['invoice_amount'] 	= add_currency( format_number( $invoice_array['total'] * $invoice_array['change_rate'] ), $invoice_array['client_id'] );
					$data['amount_paid'] 		= add_currency( format_number( $invoice_array['amount_paid'] * $invoice_array['change_rate'] ), $invoice_array['client_id'] );
					$data['amount_due'] 		= add_currency(format_number($invoice_array['amount_due'] * $invoice_array['change_rate']), $invoice_array['client_id']);
					$data['client_account'] 	= base_url( 'index.php/client/index?id=' ) . _eID( $invoice_array['client_id'] );
					$data['logo_url'] 			= base_url( 'uploads/admin/img/' ). get_option( 'logo' );
					$data['btn_client_account'] = __( 'btn_client_account', $invoice_array['client_id'] );

					$this->load->helper('file');
					$lang 					= lang_code_to_notif_string( client( $client_id, 'client_language' ) );
					$path_new_invoice 		= './notifications/' . $lang . '/invoices/new_invoice.php';
					$path_paid_invoice 		= './notifications/' . $lang . '/invoices/invoice_paid.php';
					$path_refunded_invoice 	= './notifications/' . $lang . '/invoices/invoice_refunded.php';

					$to = client( $invoice_array['client_id'], 'client_email' );
					
					$template = ( $invoice_array['invoice_status'] === "Paid" ) ? read_file( $path_paid_invoice ):
					read_file( $path_new_invoice );
					$subject = ( $invoice_array['invoice_status'] === "Paid" ) ? __( 'subject_payment_confirmation', $invoice_array['client_id'] ) : __( 'subject_invoice_generated', $invoice_array['client_id'] );
					$message = ( $invoice_array['invoice_status'] === "Paid" ) ? 
					$this->parser->parse_string( $template, $data, true ) : 
					$this->parser->parse_string( $template, $data, true );

					if ($client_id) {
						$client_cc = json_decode( null !== client($client_id, 'client_cc' ) ? client($client_id, 'client_cc' ): json_encode( array() ));
						if (1 == get_option('attach_pdf_to_invoice')) {
							generate_pdf_file('invoices', $invoice_number);
						}
					}
					if ( !_has_option('turn_off_customer_invoice_emails')) {
						send_email($to, $message, $subject, null, $client_cc, FCPATH.'uploads/pdf/invoices/'.$invoice_number.'.pdf', get_option('invoice_prefix').invoice_number_format() . sprintf("%04s", $invoice_number).'.pdf');
					}
				}

				// Update next invoice ID
				setAccountEntityId('invoice', $invoice_number);
				update_order($order_number, array('invoice_number' => $invoice_number));
			}
		}
		return false;
	}
	public function woocommerce_order_to_mmanager_order( $order_object )
	{
		$order_number = _dID($_REQUEST['order_number']);
		$order = get_order_by_number($order_number);
		$order_items = get_sale_items($order_number);

		$order_key = next_order_number();

		if ($order) {
			foreach ($order as $order) {
				// Get the change rate if multicurrency is enabled
				$currency = $order->client_id == 0 ? get_option('user_currency') : client($order->client_id, 'client_currency');
				$change_rate = _has_option('change_currency') ? get_change_rate( $currency ) : 1;


				$orderTable 								= []; // All data that will be saved on oc_orders
				$orderTable['change_rate'] 					= $change_rate; // Save change_rate for reference
				$orderTable['user_id'] 						= $m->getUserID();
				$orderTable['order_number'] 				= next_order_number();
				$orderTable['order_key'] 					= $order_key;
				$orderTable['order_status'] 				= 'Pending';
				$orderTable['client_id'] 				    = $order->client_id;
				$orderTable['date'] 						= date('Y-m-d');
				$orderTable['due_date'] 					= date('Y-m-d');
				$orderTable['subtotal'] 					= $order->subtotal;
				$orderTable['tax_amount'] 					= $order->tax_amount;
				$orderTable['global_discount'] 				= $order->global_discount;
				$orderTable['discount_amount'] 				= $order->discount_amount;
				$orderTable['total'] 						= $order->total;
				$orderTable['notes'] 						= $order->notes;
				$orderTable['subject'] 						= $order->subject;
				$orderTable['amount_paid']					= 0;
				$orderTable['amount_due']					= $order->total;
				$orderTable['credit_used']					= 0;
				$orderTable['sent']							= 'no'; // Set Invoice to sent or not

				$this->orders->set_table('orders');
				$this->orders->save($orderTable);
				$event = array(
					'action' 					=> 'add',
					'referrer' 					=> 'orders',
					'referrer_id' 				=> $this->db->insert_id(),
					'referrer_value' 			=> $order_number,
					'user_id' 					=> user('user_id'),
					'status' 					=> 'success'
				);
				record_event($event);


				foreach ($order_items as $item) {
					$itemTable 									= []; // Data that will be saved in oc_orderitems table
					$itemTable['order_number']					= $order_key;
					$itemTable['order_key']						= $order_key;
					$itemTable['item_price']					= $item->item_price;
					$itemTable['item_quantity']					= $item->item_quantity;
					$itemTable['item_discount']					= $item->item_discount;
					$itemTable['item_subtotal']					= $item->item_subtotal;
					$itemTable['item_tax_rate']					= $item->item_tax_rate;
					$itemTable['available']						= 0 < get_item_service_vars_by_name($item->item_name, 'available') ? '1' : 0;
					$itemTable['item_uom']						= $item->item_uom;
					$itemTable['item_sku']						= $item->item_sku;
					$itemTable['item_id']						= $item->item_id;
					$itemTable['item_name']						= $item->item_name;
					$itemTable['client_id']						= $item->client_id;
					$itemTable['user_id']						= $m->getUserID();

					$this->orders->set_table('orders_items');
					$this->orders->save($itemTable);
				}
			}
		}
		setAccountEntityId('order', $order_key);
	}
	/**
	 * Send Quote Notification Email
	 * @param type $quote_number 
	 * @return type mixed
	 */
	public function send_quote_email($quote_number, $context=null)
	{
		$attachments = [];
		$quotes = get_quote_by_number($quote_number);
		foreach ($quotes as $key => $quote) 
		{
			$client_language 				= client($quote->client_id, 'client_language');
			$client_id 						= $quote->client_id;
			$quote_number 					= $quote->quote_number;
			$subject 						= $quote->subject;
			$date 							= $quote->date;
			$due_date 						= $quote->due_date;
			$subtotal 						= $quote->subtotal;
			$tax_amount 					= $quote->tax_amount;
			$discount_amount 				= $quote->discount_amount;
			$total 							= $quote->total;
			$amount_due 					= $quote->amount_due;
			$quote_status 					= $quote->quote_status;
			$notes 							= $quote->notes;
			$sent 							= 'yes';

			if ($quote->attachments) {
				$attachments = json_decode($quote->attachments, true);
			}
			// Get the change rate if multicurrency is enabled
			$currency 						= $client_id == 0 ? get_option('user_currency') : client($client_id, 'client_currency');
			$change_rate 					= _has_option('change_currency') ? get_change_rate( $currency ) : 1;

			$quote_array = array(
				'client_id' 		=> $client_id,
				'quote_number' 		=> $quote_number,
				'subject' 			=> $subject,
				'date' 				=> $date,
				'due_date' 			=> $due_date,
				'subtotal' 			=> $subtotal,
				'tax_amount' 		=> $tax_amount,
				'discount_amount' 	=> $discount_amount,
				'total' 			=> $total,
				'amount_due' 		=> $amount_due,
				'quote_status' 		=> $quote_status,
				'notes' 			=> $notes,
				'sent' 				=> $sent,
				'change_rate' 		=> $change_rate
			);
			if ($client_id) {
				$data = array(
					'quote_url'		        => base_url('index.php/viewquote/view?id=') . _eID( $quote_number ),
					'accept_quote_url'	    => base_url('index.php/viewquote/view_accepted?id=') . _eID( $quote_number ),
					'decline_quote_url'	    => base_url('index.php/viewquote/view_declined?id=') . _eID( $quote_number ),
					'to_name'               => html_entity_decode(client( $client_id, 'name_company' )),
					'date'		    	    => _fdate( $client_language, $date ),
					'due_date'		        => _fdate( $client_language, $due_date ),
					'quote_status'	        => __( strtolower( $quote_status ), $client_id ),
					'quote_number' 	        => get_option( 'quote_prefix' ).invoice_number_format() . sprintf("%04s", $quote_number),
					'quotesubject'	        => $subject,
					'btn_accept_quote'      => __( 'btn_accept_quote', $client_id ),
					'btn_decline_quote' 	=> __( 'btn_decline_quote', $client_id ),
					'btn_client_account'	=> __( 'btn_client_account', $client_id )
					);

				$data['quote_amount'] 		= add_currency( format_number( $total * $change_rate ), $client_id );
				$data['amount_due'] 		= add_currency( format_number( $amount_due * $change_rate ), $client_id );
				$data['client_account'] 	= base_url('index.php/client/index?id=') . _eID( $client_id );
				$data['logo_url'] 			= base_url('uploads/admin/img/'). get_option('logo');

				$this->load->helper('file');
				$lang 					= lang_code_to_notif_string( client( $client_id, 'client_language' ) );
				$path_new_quote 		= './notifications/' . $lang . '/quotes/new_quote.php';
				$path_accepted_quote 	= './notifications/' . $lang . '/quotes/quote_accepted.php';

				$to = client( $client_id, 'client_email' );
				
				$template 	= ( $quote_status === "Accepted" ) ? read_file( $path_accepted_quote ):
				read_file( $path_new_quote );
				$subject 	= ( $quote_status === "Accepted" ) ? __( 'subject_quote_approved', $client_id ) : __( 'subject_quote_generated', $client_id );
				$message 	= ( $quote_status  === "Accepted" ) ? 
				$this->parser->parse_string( $template, $data, true ) : 
				$this->parser->parse_string( $template, $data, true );

				if ($client_id) {
					$client_cc = json_decode( null !== client($client_id, 'client_cc' ) ? client($client_id, 'client_cc' ): json_encode( array() ));
					if (1 == get_option('attach_pdf_to_invoice')) {
						generate_pdf_file('quotes', $quote_number);
					}
				}
				if ( !_has_option('turn_off_customer_quote_emails')) {
					$attachments[] = FCPATH.'uploads/pdf/quotes/'.$quote_number.'.pdf';
					send_email($to, $message, $subject, null, $client_cc, $attachments, get_option('quote_prefix').invoice_number_format() . sprintf("%04s", $quote_number).'.pdf');
				}
			}
		}
		return false;
	}

	/**
	 * Get sale data for eCharts
	 * @param type $interval 
	 * @return type mixed
	 */
	public function chartSales($interval)
	{
		$sales = [];
		$this->load->model('reports/reports_model', 'reports');
		$results = $this->reports->chartSales($interval);
		foreach ($results as $value) 
		{
			$sales[] = array(
				'name' => $value->name,
				'value' => floatval($value->value),
				'count' => $value->count
			);
		}
		return $sales;
	}
	/**
	 * Get the category ID based on item ID
	 * @param type|null $item_id 
	 * @return type int
	 */
	public function category_id($item_id=null)
	{
		if(!$item_id)
		{
			return 0;
		}
		else
		{
			$this->load->model('products/products_model', 'products');
			$groups = $this->products->category_id($item_id);
			if ($groups)
			{
				foreach ($groups as $v) {
					return $v->group_id;
				}
			}
		}
	}
	/**
	 * Get the category name for a given category ID
	 * @param type $category_id 
	 * @return type string
	 */
	public function category_name($category_id)
	{
		$this->load->model('products/products_model', 'products');
		$results = $this->products->category_name($category_id);
		if (!$results)
		{
			return __('label_uncategorized');
		}
		else
		{
			return $this->products->category_name($category_id);
		}
	}
	/**
	 * Makes payment for Mollie gateway
	 * @param type $invoice_number 
	 * @param type $transaction_id 
	 * @return type mixed
	 */
	public function mass_payment($invoice_number, $transaction_id)
	{
		return mollie_payment($invoice_number, $transaction_id);
	}
	/**
	 * Makes payment for Stripe Gateway
	 * @param type $invoice_number 
	 * @param type $transaction_id 
	 * @return type mixed
	 */
	public function mass_payment_stripe($invoice_number, $transaction_id)
	{
		return stripe_payment($invoice_number, $transaction_id);
	}
	/**
	 * Makes payment for Ingenico gateway
	 * @param type $invoice_number 
	 * @param type $transaction_id 
	 * @return type mixed
	 */
	public function mass_payment_ingenico($invoice_number, $transaction_id)
	{
		return ingenico_payment($invoice_number, $transaction_id);
	}
	/**
	 * Marks a invoice refund and notify customer
	 * @param type $invoice_number 
	 * @param type|null $amount 
	 * @return type mixed
	 */
	public function mark_refunded($invoice_number, $amount=null)
	{
		mark_refunded($invoice_number, $amount);
		$this->send_invoice_email($invoice_number);
	}
	/**
	 * Get current change rate for a given currency
	 * @param type $currency 
	 * @param type|null $base 
	 * @return type float
	 */
	public function get_change_rate($currency, $base = null)
	{
		$change_rate = 1;
		if ( _has_option('change_currency') )
		{
			if(!$base)
			{
				$base = get_option('user_currency');
			}
			$currency_pair = $base . '/' . $currency;

			try {
				$swap = (new Builder())
				    ->add('central_bank_of_czech_republic')
				    ->add('central_bank_of_republic_turkey')
				    ->add('european_central_bank')
				    ->add('google')
				    ->add('national_bank_of_romania')
				    ->add('webservicex')
				    ->add('yahoo')
				    ->add('webservicex')
				    ->add('russian_central_bank')
				    ->add('cryptonator')
				    ->build();
				$change_rate = $swap->latest($currency_pair)->getValue();
			} catch (Exception $e) {
				$change_rate = 0;
			}
			if ( ! $change_rate) {

				set_option(array(
	                'option_name' => "change_currency",
	                'option_value' => 0
	                )
				);
				update_clients_currency();
				update_providers_currency();
				$change_rate = 1;
			}
			return $change_rate;
		}
		return $change_rate; 
	}
	/**
	 * Initialize the class
	 * @return type mixed
	 */
	private function _init()
	{
		$payment_methods = [];
		$p_methods = get_payment_methods();
		
		$inactiveItems = null == get_items('inactive') ? []: get_items('inactive');
		$inactiveServices = null == get_services('inactive') ? []: get_services('inactive');

		if ($p_methods)
		{
			foreach ($p_methods as $v) {
				array_push($payment_methods, $v->payment_method_name);
			}
		}
		if (get_option('app_url') == null) 
		{
			set_option(array('option_name' => 'app_url', 'option_value' => base_url()));
		}
		$user_timezone = (!get_option('user_timezone')) ? 'Europe/Paris' : get_option('user_timezone');
		$data = array(
			'base_url' => $this->config->item('base_url'),
			'default_duedate' => (!get_option('default_duedate')) ? '0': get_option('default_duedate'),
			'avatarPath' => 'uploads/users/avatars/',
			'copyright'	=> copyright_info(),
			'app_title' => "m'Manager | Invoices Management System",
			'langMeta' => language_meta(),
			'app_name' => "m'Manager | Invoices Management System",
			'username' => user('first_name'),
			'user_avatar' => get_user_avatar(),
			'user_id' => $this->_userID,
			'countries' => get_countries(),
			'user_language' => $this->user_language,
			'user_timezone' => (!get_option('user_timezone')) ? 'Europe/Paris' : get_option('user_timezone'),
			'app_description' => "m’Manager is a small but full featured piece of software designed for freelancers and based on CodeIgniter 3. It will help you manage easily your business, create and send automatically your invoices and estimates, keep track of your turnover, tax and items sale performances. Many cool features are coming soon... So welcome on board ! You will enjoy it !",
			'app_author' => "Eric Claver AKAFFOU",
			'currencies' => get_currencies(),
			'roles' => get_roles(),
			'timezones' => get_timezone(),
			'system_version' => ONERIC_VERSION,
			'money_preferences' => money_preferences(),
			'payment_methods' => $payment_methods,
			'additional_taxes' => get_additional_taxes(),
			'additional_tax_criteria' => $payment_methods,
			'countOverdueInvoices' => null == count_invoices('Overdue') ? '' : count_invoices('Overdue'),
			'count_uncompleted_sales' => null == count_sale_orders() ? '' : count_sale_orders(),
			'count_clients'			=> null == count_clients('active') ? '': count_clients('active'),
			'count_items'			=> 0 == count($inactiveItems) ? '' : count($inactiveItems),
			'count_services'			=> 0 == count($inactiveServices) ? '' : count($inactiveServices),
			'number_format' => array(
				'1' => '123,456.78',
				'2' => '123.456,78',
				'3' => '123 456.78',
				'4' => '123 456,78',
				'5' => '123.456',
				'6' => '123 456'
				),
			'currency_placement' => array(
				'before' => __('before') .  ' €123',
				'after' => __('after') . ' 123€',
				'none' => __('none') . ' 123'
				),
			'languages' => array(
				'en_GB' => 'English-GB',
				'en_US' => 'English',
				'nl_NL' => 'Dutch',
				'fr_FR' => 'Français',
				'it_IT' => 'Italian',
				'ar_AR' => 'اللغة العربية',
				'el_EL' => 'Ελληνικά',
				'es_ES'	=> 'Español',
				'ro'	=> 'Română',
				'sv_SE'	=> 'Svenska'
				),
			'invoice_numbering' => array(
				'1' => '1, 2, 3...',
				'2' => date('Y') . '#0001',
				'3' => date('Y/m') . '#0001',
				'4' => date('y') . '0001'
				),
			'time_format' => array(
				'%h:%i %a' => mdate('%h:%i %a', now($user_timezone)),
				'%H:%i' => mdate('%H:%i', now($user_timezone))
				),
			'date_format' => array(
				'j F Y' => strftime( '%d %B %Y', strtotime( date("Y-m-d") ) ),
				'F j, Y' => strftime( '%B %d, %Y', strtotime( date("Y-m-d") ) ),
				'Y-m-d' => date("Y-m-d"),
				'm/d/Y' => date("m/d/Y"),
				'd/m/Y' => date("d/m/Y")
				),
			'appointment_types' => (object) array(
				'1' => array('id' => 1, 'value' => __('label_in_shop')),
				'2' => array('id' => 2, 'value' => __('label_onsite')),
				'3' => array('id' => 3, 'value' => __('label_phone_call')),
			),
			'assets_types' => (object) array(
				'1' => array('id' => 1, 'value' => __('label_computers')),
				'2' => array('id' => 2, 'value' => __('label_mobile_devices')),
				'3' => array('id' => 3, 'value' => __('label_other_electronics')),
				'4' => array('id' => 4, 'value' => __('label_mechanics')),
			),
			'invoice_numb_prefixe' => invoice_number_format(),
			'customer_currency' => json_decode( null !== get_option( 'customer_currency' ) ? get_option( 'customer_currency' ): json_encode( array( get_option( 'user_currency' ) ) ) ),
			'eligible_roles' => json_decode( null !== get_option( 'eligible_roles' ) ? get_option( 'eligible_roles' ): json_encode([])),
			'notify_when_invoice_is' => json_decode( null !== get_option( 'notify_when_invoice_is' ) ? get_option( 'notify_when_invoice_is' ): json_encode([])),
			'pc' => get_option('purchase_code'),
			'd' => get_options(),
			'taxes' => get_tax()
			);
		$this->load->vars($data);
		
		upgrade_db_add_mpn();
		db_add_shipment_table();
		db_add_overdue_invoice_reminder();
		db_add_item_serial_number();
		db_add_time_tracker();
		db_add_proforma();
		db_add_edit_item_serial_number_col_type();
		db_add_notifications();
		addAnnualNumbering();
		if ( floatval(get_option('stable_version')) < 2.3 )
		{
			db_upgrade_decimal_precision();
		}
		db_add_invoice_late_fees();
		db_add_invoices_attachments();
		//Recreate late fees record if deleted by mistake
		_has_late_fees_item();
	}
	/**
	 * Helper function to set user globals
	 * @return type mixed
	 */
	private function _setUserVars()
	{
		$this->_userID = user('user_id');
		$this->_userEmail = user('email');
		$this->_userFname = user('first_name');
		$this->_userCompany = user('company');
		$this->_userAddress = user('address');
	}
	/**
	 * Helper function to set global options
	 * @return type
	 */
	private function _setOption()
	{
		$this->load->model('settings/settings_model', 'settings');
		$this->settings->set_table('users_options');

		if ($this->_userID)
		{
			$where = 'user_id = ' . $this->_userID;
			$options = $this->settings->get_where($where);

			if ($options)
			{
				foreach ($options as $option) 
				{
					$this->_userOptions[$option->option_key] = $option->option_value;
					$this->_optionsID[$option->option_key] = $option->option_id;
				}
				return $this;
			}
		}
		return FALSE;
	}
}